/*
  CDXSLoader by Salvatore Russo (October 2007)

  See the header file for additional information including use and distribution rights.
*/

#include "CDXSLoader.h"
#ifndef DXS_EXTERNAL
#include "os.h"
#else
#undef DXS_READER_DEBUG
//#include "CDXSParameters.h"
#endif
//#include <string.h>
//#include "IXMLReader.h"
//#include "IDummyTransformationSceneNode.h"
#include "SAnimatedMesh.h"
//#include "fast_atof.h"
//#include "quaternion.h"
//#include "ILightSceneNode.h"
//#include "ICameraSceneNode.h"
//#include "IMeshManipulator.h"
//#include "IAttributes.h"
//#include "IMeshCache.h"
#include "IMeshSceneNode.h"
#include "SMeshBufferLightMap.h"
#include "dxssupport.h"




namespace irr
{
    namespace scene
    {
        const core::stringc dxsSceneName = "scene";
        const core::stringc dxsSettingsName = "settings";
        const core::stringc dxsMaterialsName = "materials";
        const core::stringc dxsCategoryName = "category";
        const core::stringc dxsMaterialName = "material";
        const core::stringc dxsLayerName = "layer";
        const core::stringc dxsTextureName = "texture";
        const core::stringc dxsColorName = "color";
        const core::stringc dxsLightmapName = "lightmap";
        const core::stringc dxsPrimitivesName = "primitives";
        const core::stringc dxsPrimitiveName = "primitive";
        const core::stringc dxsTagName = "tag";
        const core::stringc dxsVerticesName = "vertices";
        const core::stringc dxsVertexName = "vertex";
        const core::stringc dxsPolygonsName = "polygons";
        const core::stringc dxsPolyName = "poly";
        const core::stringc dxsLightsName = "lights";
        const core::stringc dxsLightName = "light";
        const core::stringc dxsAmbientName = "ambient";
        const core::stringc dxsFalse = "false";
        const core::stringc dxsTrue = "true";

        //blending type not really used except for alphablending
        const core::stringc dxsBlendAlpha = "alphablend";
        const core::stringc dxsBlendReplace = "replace";
        const core::stringc dxsBlendAdd = "add";
        const core::stringc dxsBlendModulate = "modulate";



        /** Constructor*/
        CDXSLoader::CDXSLoader(video::IVideoDriver* driver, ISceneManager* smgr, io::IFileSystem* fs)
                : Driver(driver) , SceneManager(smgr) , FileSystem(fs) , DummyMesh(0)
        {
#ifdef _DEBUG
            IUnknown::setDebugName("CDXSLoader");
#endif
        }


        /** Destructor*/
        CDXSLoader::~CDXSLoader()
        {
            if (DummyMesh)
                DummyMesh->drop();
        }



        /** Given first three points of a face, returns a face normal*/
        void CDXSLoader::GetFaceNormal(f32 a[3], //First point
                                       f32 b[3], //Second point
                                       f32 c[3], //Third point
                                       f32 out[3]) //Normal computed
        {
            f32 v1[3], v2[3];

            v1[0] = a[0] - b[0];
            v1[1] = a[1] - b[1];
            v1[2] = a[2] - b[2];

            v2[0] = b[0] - c[0];
            v2[1] = b[1] - c[1];
            v2[2] = b[2] - c[2];

            out[0] = (v1[1] * v2[2]) - (v1[2] * v2[1]);
            out[1] = (v1[2] * v2[0]) - (v1[0] * v2[2]);
            out[2] = (v1[0] * v2[1]) - (v1[1] * v2[0]);

            f32 dist = (f32)sqrtf((out[0] * out[0]) + (out[1] * out[1]) + (out[2] * out[2]));

            if (dist == 0.0f)
                dist = 0.001f;

            out[0] /= dist;
            out[1] /= dist;
            out[2] /= dist;
        }


        /**Creates/loads an animated mesh from the file.
         \return Pointer to the created mesh. Returns 0 if loading failed.
         If you no longer need the mesh, you should call IAnimatedMesh::drop().
         See IUnknown::drop() for more information.*/
        IAnimatedMesh* CDXSLoader::createMesh(io::IReadFile* file)
        {

            io::IXMLReaderUTF8* reader = FileSystem->createXMLReaderUTF8(file);
            if (!reader)
                return 0;
            //set DXS parent node
            core::stringc node_name=SceneManager->getParameters()->getAttributeAsString(DXS_ROOT_NODE_NAME);
            parent=SceneManager->addEmptySceneNode(0);

            CurrentlyLoadingMesh = file->getFileName();
            //base node of DeleD scene is filename or the one selected
            if(node_name!="")
                parent->setName(node_name.c_str());
            else
                parent->setName(CurrentlyLoadingMesh.c_str());

            // read until scene section, skip other parts
            while(reader->read())
            {
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsSceneName == reader->getNodeName())
                        readSceneSection(reader); //read scene
                    else
                        skipSection(reader, true); // unknown section
                }
            }

            reader->drop();

            scene::IAnimatedMesh* returnMesh = DummyMesh;

            // because this loader loads and creates a complete scene instead of
            // a single mesh, return an empty dummy mesh to make the scene manager
            // know that everything went well.
            if (!DummyMesh)
            {
                DummyMesh = new SAnimatedMesh();
                returnMesh = DummyMesh;
            }

            returnMesh->grab(); // store until this loader is destroyed

            DummyMesh->drop();
            DummyMesh = 0;

            return returnMesh;
        }


        //! skips an (unknown) section in the dxs document
        void CDXSLoader::skipSection(io::IXMLReaderUTF8* reader, bool reportSkipping)
        {
#ifndef DXS_READER_DEBUG
            if (reportSkipping) // always report in DXS_READER_DEBUG mode
#endif

                #ifndef DXS_EXTERNAL
                os::Printer::log("DXS skipping section", core::stringc(reader->getNodeName()).c_str());
                #endif

            // skip if this element is empty anyway.
            if (reader->isEmptyElement())
                return;

            // read until we've reached the last element in this section
            int tagCounter = 1;

            while(tagCounter && reader->read())
            {
                if (reader->getNodeType() == io::EXN_ELEMENT &&
                        !reader->isEmptyElement())
                {
#ifdef DXS_READER_DEBUG
                    if (reportSkipping)
                        os::Printer::log("DXS unknown element:", core::stringc(reader->getNodeName()).c_str());
#endif // DXS_READER_DEBUG

                    ++tagCounter;
                }
                else
                    if (reader->getNodeType() == io::EXN_ELEMENT_END)
                        --tagCounter;
            }
        }




        //! reads the <scene> section and its content if its version is >= 1.6
        void CDXSLoader::readSceneSection(io::IXMLReaderUTF8* reader)
        {
            //this is just first released version so if < return
            if (reader->isEmptyElement() && reader->getAttributeValueAsFloat("version")<1.6)
                return;

            //read fundamental elements (no lights for the moment)
            while(reader->read())
            {
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsSettingsName == reader->getNodeName())
                        readSettingsSection(reader);
                    else
                        if (dxsMaterialsName == reader->getNodeName())
                            readMaterialsSection(reader);
                        else
                            if (dxsPrimitivesName == reader->getNodeName())
                                readPrimitivesSection(reader);
                            else
                                if (dxsLightsName == reader->getNodeName())
                                    skipSection(reader, true);
                                else
                                    skipSection(reader, true); // unknown section*/
                }
            }
        }

        //! reads the <settings> section and its content
        void CDXSLoader::readSettingsSection(io::IXMLReaderUTF8* reader)
        {
            //read all settings also if not used

            Settings.scene_name=reader->getAttributeValue("name");
            Settings.scene_author=reader->getAttributeValue("author");
            Settings.scene_comments=reader->getAttributeValue("comments");
            Settings.shadowOpacity=reader->getAttributeValueAsFloat("shadowOpacity");

#ifdef DXS_READER_DEBUG

            os::Printer::log("DeleD DXS: reading settings...");
            os::Printer::log("DeleD DXS: scene name", Settings.scene_name.c_str());
            os::Printer::log("DeleD DXS: scene author", Settings.scene_author.c_str());
            os::Printer::log("DeleD DXS: scene comments", Settings.scene_comments.c_str());
#endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsSettingsName == reader->getNodeName()) && reader->read())//;//)
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsAmbientName == reader->getNodeName())
                        Settings.ambientColor=video::SColor(255,reader->getAttributeValueAsInt("r"),reader->getAttributeValueAsInt("g"),reader->getAttributeValueAsInt("b"));
                    else
                        skipSection(reader, true); // unknown section
                }
            SceneManager->setAmbientLight(Settings.ambientColor);
        }

        //! reads the <materials> section and its content
        void CDXSLoader::readMaterialsSection(io::IXMLReaderUTF8* reader)
        {
#ifdef DXS_READER_DEBUG
            os::Printer::log("DeleD DXS: reading materials...",);
#endif
            //first of all get path info
            String path = "";
            StringList filepath,filepath1;
            if ( !SceneManager->getParameters()->existsAttribute(DXS_TEXTURE_PATH) )
            {
                //get the right path for textures
                filepath = SubdivideString(String(CurrentlyLoadingMesh),"\\");
                filepath.erase(filepath.size()-1);
                filepath1 = SubdivideString(String(CurrentlyLoadingMesh),"/");
                filepath1.erase(filepath1.size()-1);
#ifdef DXS_READER_DEBUG

                os::Printer::log("DeleD DXS: textures path doesn't exist",CurrentlyLoadingMesh.c_str());
#endif

            }
            else
            {
                filepath = SubdivideString(String( SceneManager->getParameters()->getAttributeAsString(DXS_TEXTURE_PATH)),"\\");
                filepath1 = SubdivideString(String( SceneManager->getParameters()->getAttributeAsString(DXS_TEXTURE_PATH)),"/");
#ifdef DXS_READER_DEBUG

                os::Printer::log("DeleD DXS: textures path exists",path.c_str());
#endif

            }
            if(filepath1.size()>filepath.size())
            {
                filepath.clear();
                filepath=filepath1;
                filepath1.clear();
            }

            for (int j=0; j<(int)(filepath.size()); j++)
                path = path + filepath[j] + String("\\");

#ifdef DXS_READER_DEBUG

            os::Printer::log("DeleD DXS: textures path",path.c_str());
#endif

            //read material category
            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsMaterialsName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsCategoryName == reader->getNodeName())
                        readCategorySection(reader,path);
                    else
                        skipSection(reader, true); // unknown section
                }
        }

        //! reads the <category> section and its content
        void CDXSLoader::readCategorySection(io::IXMLReaderUTF8* reader,core::stringc path)
        {
            core::stringc category=reader->getAttributeValue("name");
#ifdef DXS_READER_DEBUG

            os::Printer::log("DeleD DXS: reading materials category",category.c_str());
#endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsCategoryName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsMaterialName == reader->getNodeName() && dxsTrue==reader->getAttributeValue("used"))
                        readMaterialSection(reader,category,path);
                    else
                        skipSection(reader, true); // unknown section
                }
        }

        //! reads the <material> section and its content
        void CDXSLoader::readMaterialSection(io::IXMLReaderUTF8* reader,core::stringc category,core::stringc path)
        {
            SDXSMaterial material;
            material.Name=reader->getAttributeValue("name");
            material.Category=category;
            material.Id=reader->getAttributeValueAsInt("id");
            material.Category=category;
            material.Mat=video::SMaterial();
            material.LayersNum=0;
#ifdef DXS_READER_DEBUG

            os::Printer::log("DeleD DXS: reading material",material.Name.c_str());
#endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsMaterialName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsLayerName == reader->getNodeName())
                        readLayerSection(reader,&material,path);
                    else
                        skipSection(reader, true); // unknown section
                }

            // add material to list of loaded materials positioning lightmap as second texture
            if(material.LayersNum>=2)
            {
                int index=0;
                material.LightMapNum=0;
                for(int i=0;i<material.LayersNum;i++)
                {
                    if(material.Type[i]==LIGHTMAP)
                    {
                        index=i;
                        material.LightMapNum=i;
                    }
                }
                if(index!=0 && index!=1)
                {
                    video::ITexture *tex=material.Mat.Textures[index];
                    material.Mat.Textures[index]=material.Mat.Textures[1];
                    material.Mat.Textures[1]=tex;
                    core::stringc tex_s=material.Blend[index];
                    material.Blend[index]=material.Blend[1];
                    material.Blend[1]=tex_s;
                    LayerType type=material.Type[index];
                    material.Type[index]=material.Type[1];
                    material.Type[1]=type;
                }
            }
            Materials.push_back(material);
        }

        //! reads the <layer> section and its content
        void CDXSLoader::readLayerSection(io::IXMLReaderUTF8* reader,SDXSMaterial *material,core::stringc path)
        {
            material->Blend.push_back(reader->getAttributeValue("blend"));
            LayerType tipo;
            if(dxsLightmapName==reader->getAttributeValue("type"))
                tipo=LIGHTMAP;
            else if(dxsTextureName==reader->getAttributeValue("type"))
                tipo=TEXTURE;
            else if(dxsColorName==reader->getAttributeValue("type"))
                tipo=COLOR;
#ifdef DXS_READER_DEBUG

            os::Printer::log("DeleD DXS: reading layer of type",reader->getAttributeValue("type").c_str());
#endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsLayerName == reader->getNodeName()) &&reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsTextureName == reader->getNodeName())
                    {

                        material->Type.push_back(tipo);
                        Driver->setTextureCreationFlag  (ETCF_ALWAYS_32_BIT,true);
                        core::stringc texturename=reader->getAttributeValue("file");
                        if(!SceneManager->getParameters()->existsAttribute(DXS_USE_MATERIALS_DIRS) || !SceneManager->getParameters()->getAttributeAsBool(DXS_USE_MATERIALS_DIRS))
                        {
                            StringList texts=SubdivideString(String(texturename),String("\\"));
                            texturename=texts[texts.size()-1];
                            texts.clear();
                        }
#ifdef DXS_READER_DEBUG
                        if(tipo==TEXTURE)
                            os::Printer::log("DeleD DXS: reading texture",(path+texturename).c_str());
                        else
                            os::Printer::log("DeleD DXS: reading lightmap",(path+texturename).c_str());
#endif

                        material->Mat.Textures[material->LayersNum]=Driver->getTexture((path+texturename).c_str());
                        material->LayersNum++;

                    }
                    else if(dxsColorName == reader->getNodeName())
                    {
                        material->Type.push_back(tipo);

                        SColor color(reader->getAttributeValueAsInt("a"),
                                     reader->getAttributeValueAsInt("r"),
                                     reader->getAttributeValueAsInt("g"),
                                     reader->getAttributeValueAsInt("b"));
                        core::stringc color_name("Color_");
                        color_name=color_name+reader->getAttributeValue("a");
                        color_name=color_name+reader->getAttributeValue("r");
                        color_name=color_name+reader->getAttributeValue("g");
                        color_name=color_name+reader->getAttributeValue("b");

                        s32 col = color.color;
                        s32 buf[64];
                        for (int k=0; k<64; k++)
                            buf[k]=col;

                        //just for compatibility with older Irrlicht versions
                        //to support transparent materials
                        if(color.getAlpha()!=255 && material->Blend[material->Blend.size()-1]==core::stringc("alphablend"))
                            Driver->setTextureCreationFlag  (ETCF_ALWAYS_32_BIT,true);
                        IImage *immagine=Driver->createImageFromData(ECF_A8R8G8B8,
                                         core::dimension2d<s32>(8,8),buf);
#ifdef DXS_READER_DEBUG

                        os::Printer::log("DeleD DXS: reading color",color_name.c_str());
#endif

                        material->Mat.Textures[material->LayersNum]=Driver->addTexture(color_name.c_str(), immagine);
                        material->LayersNum++;
                        immagine->drop();

                    }
                    else
                        skipSection(reader, true); // unknown section
                }

        }

        //! reads the <primitives> section and its content
        void CDXSLoader::readPrimitivesSection(io::IXMLReaderUTF8* reader)
        {
#ifdef DXS_READER_DEBUG
            os::Printer::log("DeleD DXS: reading primitives...");
#endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsPrimitivesName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsPrimitiveName == reader->getNodeName())
                        readPrimitiveSection(reader);
                    else
                        skipSection(reader, true); // unknown section
                }
        }


        //! reads the <primitive> section and its content
        void CDXSLoader::readPrimitiveSection(io::IXMLReaderUTF8* reader)
        {

            CurrVertices.clear();
            CurrPolys.clear();
            SMesh *Mesh=new SMesh();
            core::stringc MeshName=reader->getAttributeValue("name");
            s32 MeshId=reader->getAttributeValueAsInt("id");
            #ifdef DXS_READER_DEBUG
            os::Printer::log("DeleD DXS: reading primitive",MeshName.c_str());
            #endif

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsPrimitiveName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsVerticesName == reader->getNodeName())
                        readVertexSection(reader);
                    else if (dxsPolygonsName == reader->getNodeName())
                        readPolySection(reader);
                    else
                        skipSection(reader, true); // unknown section
                }

            //create a meshbuffer for each material, then we'll remove empty ones
            for (u32 i=0; i<Materials.size(); i++)
            {
                //create a new SMeshBufferLightMap for each material
                SMeshBufferLightMap* buffer = new SMeshBufferLightMap();
                buffer->Material=Materials[i].Mat;
                buffer->Material.MaterialType = video::EMT_LIGHTMAP_LIGHTING  ;
                buffer->Material.Wireframe = false;
                buffer->Material.Lighting = true;
                buffer->Material.BilinearFilter = true;
                Mesh->addMeshBuffer(buffer);
                buffer->drop();
            }

            // Build the mesh buffers
            for (u32 i = 0; i < CurrPolys.size(); i++)
            {
                if (CurrPolys[i].Vid.size() < 3)
                    continue;

                f32 normal[3];

                GetFaceNormal(CurrVertices[CurrPolys[i].Vid[0]].GetPos(),
                              CurrVertices[CurrPolys[i].Vid[1]].GetPos(), CurrVertices[CurrPolys[i].Vid[2]].GetPos(), normal);

                SMeshBufferLightMap * meshBuffer = 0;
                u32 MaterialId=0;
                for(u32 k=0;k<Materials.size();k++)
                {
                    if(Materials[k].Id==CurrPolys[i].Mid)
                    {
                        meshBuffer=(SMeshBufferLightMap*)Mesh->getMeshBuffer(k);
                        MaterialId=k;
                        break;
                    }
                }
                u32 base = meshBuffer->Vertices.size();

                // Add this face's verts
                u32 v;
                u32 uvsize=Materials[MaterialId].LayersNum;
                for (v = 0; v < CurrPolys[i].Vid.size() ; v++)
                {
                    //dmfVert * vv = &verts[faces[i].firstVert + v];
                    video::S3DVertex2TCoords vert(CurrVertices[CurrPolys[i].Vid[v]].x, CurrVertices[CurrPolys[i].Vid[v]].y, -CurrVertices[CurrPolys[i].Vid[v]].z,
                                                  normal[0], normal[1], normal[2], video::SColor(0,255,255,255), 0.0f, 0.0f);
                    if ( Materials[MaterialId].Blend[0]==dxsBlendAlpha &&
                            SceneManager->getParameters()->getAttributeAsBool(DXS_FLIP_ALPHA_TEXTURES))
                    {
                        u32 ind=v*uvsize;
                        vert.TCoords.set(CurrPolys[i].Uv[ind].u,-CurrPolys[i].Uv[ind].v);
                        if(uvsize>1)
                            vert.TCoords2.set(CurrPolys[i].Uv[ind+1].u,CurrPolys[i].Uv[ind+1].v);
                        else
                            vert.TCoords2.set(0,0);
                    }
                    else
                    {
                        u32 ind=v*uvsize;
                        vert.TCoords.set(CurrPolys[i].Uv[ind].u,CurrPolys[i].Uv[ind].v);
                        if(uvsize>1)
                            vert.TCoords2.set(CurrPolys[i].Uv[ind+1].u,CurrPolys[i].Uv[ind+1].v);
                        else
                            vert.TCoords2.set(0,0);
                    }
                    meshBuffer->Vertices.push_back(vert);
                }

                // Now add the indices
                // This weird loop turns convex polygons into triangle strips.
                // I do it this way instead of a simple fan because it usually
                // looks  a lot better in wireframe, for example.
                u32 h = CurrPolys[i].Vid.size() - 1, l = 0, c; // High, Low, Center
                for (v = 0; v < CurrPolys[i].Vid.size() - 2; v++)
                {
                    if (v & 1)
                        c = h - 1;
                    else
                        c = l + 1;

                    meshBuffer->Indices.push_back(base + h);
                    meshBuffer->Indices.push_back(base + l);
                    meshBuffer->Indices.push_back(base + c);

                    if (v & 1)
                        h--;
                    else
                        l++;
                }
            }

            for (u32 i=0; i<Materials.size(); i++)
            {
                //create a new SMeshBufferLightMap for each material
                SMeshBufferLightMap* buffer = (SMeshBufferLightMap*)Mesh->getMeshBuffer(i);
                if(Materials[i].Blend[0]==dxsBlendAlpha)
                {
                    buffer->Material.MaterialType = video::EMT_TRANSPARENT_ALPHA_CHANNEL;
                    buffer->Material.MaterialTypeParam =SceneManager->getParameters()->getAttributeAsFloat(DXS_ALPHA_CHANNEL_REF);
                    }
                else if(Materials[i].Type.size()==1 || Materials[i].Type[1]!=LIGHTMAP)
                {
                    buffer->Material.MaterialType = video::EMT_SOLID;
                    float mult = 100.0f - Settings.shadowOpacity;
                    mult = mult/100.0f;
                    buffer->Material.AmbientColor=Settings.ambientColor.getInterpolated  (  SColor(255,0,0,0),mult);
                   }
                else
                {
                    buffer->Material.MaterialType = video::EMT_LIGHTMAP_LIGHTING;
                   }
            }
            // delete all buffers without geometry in it.
            i = 0;
            while(i < Mesh->MeshBuffers.size())
            {
                if (Mesh->MeshBuffers[i]->getVertexCount() == 0 ||
                        Mesh->MeshBuffers[i]->getIndexCount() == 0 ||
                        Mesh->MeshBuffers[i]->getMaterial().Textures[0] == 0)
                {
                    // Meshbuffer is empty -- drop it
                    Mesh->MeshBuffers[i]->drop();
                    Mesh->MeshBuffers.erase(i);
                }
                else
                {
                    i++;
                }
            }
            for (i = 0; i < Mesh->MeshBuffers.size(); i++)
            {
                ((SMeshBufferLightMap*)Mesh->MeshBuffers[i])->recalculateBoundingBox();
            }
            Mesh->recalculateBoundingBox();
            ISceneNode *node=0;
            if(SceneManager->getParameters()->getAttributeAsBool(DXS_LOAD_AS_OCTTREE))
                node=SceneManager->addOctTreeSceneNode (Mesh, parent,MeshId);
            else
                node=SceneManager->addMeshSceneNode(Mesh, parent,MeshId);
            node->setName(MeshName.c_str());
             #ifdef DXS_READER_DEBUG
             os::Printer::log("Added DeleD DXS Node Mesh", MeshName.c_str());
             #endif
            Mesh->drop();
        }

    //! reads the <vertex> section
        void CDXSLoader::readVertexSection(io::IXMLReaderUTF8* reader)
        {


            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsVerticesName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsVertexName == reader->getNodeName())
                    {
                        SDXSVertex v;
                        v.Id=reader->getAttributeValueAsInt("id");
                        v.x=reader->getAttributeValueAsFloat("x");
                        v.y=reader->getAttributeValueAsFloat("y");
                        v.z=reader->getAttributeValueAsFloat("z");
                        CurrVertices.push_back(v);
                        #ifdef DXS_READER_DEBUG
                        os::Printer::log("DeleD DXS: reading vertex", reader->getAttributeValue("id").c_str());
                        #endif
                    }
                    else
                        skipSection(reader, true); // unknown section
                }
        }
    //! reads the <poly> section
        void CDXSLoader::readPolySection(io::IXMLReaderUTF8* reader)
        {
            //read all settings also if not used
            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsPolygonsName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsPolyName == reader->getNodeName())
                    {
                        SDXSPoly p;
                        p.Mid=reader->getAttributeValueAsInt("mid");
                        readPolyVertexSection(reader,&p);
                        CurrPolys.push_back(p);
                        #ifdef DXS_READER_DEBUG
                        os::Printer::log("DeleD DXS: reading poly");
                        #endif

                    }
                    else
                        skipSection(reader, true); // unknown section
                }
        }
//! reads the <vertex> section inside <poly> section
        void CDXSLoader::readPolyVertexSection(io::IXMLReaderUTF8* reader,SDXSPoly *poly)
        {

            while(!(reader->getNodeType() == io::EXN_ELEMENT_END && dxsPolyName == reader->getNodeName()) && reader->read())
                if (reader->getNodeType() == io::EXN_ELEMENT)
                {
                    if (dxsVertexName == reader->getNodeName())
                    {
                        u32 mat_i=0;
                        for(u32 i=0;i<Materials.size();i++)
                        {
                            if(Materials[i].Id==poly->Mid)
                            {
                                mat_i=i;
                                break;
                            }
                        }
                        core::stringc u("u");
                        core::stringc v("v");
                        s32 s_l=0;
                        if(Materials[mat_i].LightMapNum >= 1)
                            s_l=Materials[mat_i].LightMapNum;
                        poly->Vid.push_back(reader->getAttributeValueAsInt("vid"));
                        for(s32 i=0;i<Materials[mat_i].LayersNum;i++)
                        {
                            SDXSUv uv;
                            if(i==1 && s_l!=0)
                            {
                                uv.u=reader->getAttributeValueAsFloat((u+core::stringc(s_l)).c_str());
                                uv.v=reader->getAttributeValueAsFloat((v+core::stringc(s_l)).c_str());

                            }
                            else if(s_l!=0 && i==s_l)
                            {
                                uv.u=reader->getAttributeValueAsFloat((u+core::stringc(1)).c_str());
                                uv.v=reader->getAttributeValueAsFloat((v+core::stringc(1)).c_str());
                            }
                            else
                            {
                                uv.u=reader->getAttributeValueAsFloat((u+core::stringc(i)).c_str());
                                uv.v=reader->getAttributeValueAsFloat((v+core::stringc(i)).c_str());
                            }
                            poly->Uv.push_back(uv);
                        }
                    }
                    else
                        skipSection(reader, true); // unknown section
                }
        }

        /** \brief Tell us if this file is able to be loaded by this class
         based on the file extension (e.g. ".bsp")
         \return true if file is loadable.*/
        bool CDXSLoader::isALoadableFileExtension(const c8* filename)
        {
            return strstr(filename, ".dxs") != 0;
        }



    } // end namespace scene
} // end namespace irr
